# 로컬 미디어 파일을 선택하고 Whisper(Turbo)로 텍스트로 변환한 후
# 최신 Gemini 3 (Flash Preview) API를 사용하여 5줄로 요약하는 프로그램
# 업데이트: 2026-01-03

import os
import sys

# [설정] 불필요한 로그 및 경고 메시지 숨기기 (반드시 다른 import보다 상단에 위치)
os.environ['GRPC_VERBOSITY'] = 'ERROR'
os.environ['GLOG_minloglevel'] = '2'

import torch
import whisper
import tkinter as tk
from tkinter import filedialog
import google.generativeai as genai
from datetime import datetime

# ==========================================
# 사용자 설정 영역
# ==========================================
API_KEY = 'api 키 입력'  # Gemini API 키를 입력하세요

# 2026년 최신 모델 적용 (User Requested)
GEMINI_MODEL_NAME = 'gemini-3-flash-preview' 

# Whisper 모델 (large-v3-turbo가 속도/정확도 밸런스 최상)
WHISPER_MODEL_NAME = 'turbo' 
# ==========================================

def select_media_file():
    """파일 선택 대화상자 표시"""
    root = tk.Tk()
    root.withdraw()
    
    file_types = [
        ("미디어 파일", "*.mp3 *.mp4 *.wav *.m4a *.avi *.mkv *.webm"),
        ("모든 파일", "*.*")
    ]
    
    file_path = filedialog.askopenfilename(
        title="텍스트 변환할 미디어 파일 선택",
        filetypes=file_types
    )
    return file_path

def transcribe_audio(audio_path, output_path=None):
    """Whisper Turbo 모델로 STT 수행"""
    device = "cuda" if torch.cuda.is_available() else "cpu"
    print(f"🔄 처리 장치(Device): {device}")
    
    try:
        print(f"🚀 Whisper '{WHISPER_MODEL_NAME}' 모델 로드 중...")
        model = whisper.load_model(WHISPER_MODEL_NAME, device=device)
        
        print(f"🎧 '{os.path.basename(audio_path)}' 변환 시작...")
        result = model.transcribe(audio_path)
        transcribed_text = result["text"]
        
        if output_path:
            with open(output_path, "w", encoding="utf-8") as f:
                f.write(transcribed_text)
            print(f"💾 원본 텍스트 저장 완료: {output_path}")
        
        # 메모리 즉시 해제
        del model
        if device == "cuda":
            torch.cuda.empty_cache()
            
        return transcribed_text
        
    except Exception as e:
        print(f"❌ 오디오 변환 오류: {e}")
        return None

def summarize_text(text):
    """Gemini 3 Flash Preview 모델로 요약 수행"""
    try:
        genai.configure(api_key=API_KEY)
        model = genai.GenerativeModel(GEMINI_MODEL_NAME)
        
        prompt = f"""
        다음 텍스트를 분석하여 핵심 내용을 5줄로 명확하게 요약해줘.
        
        [텍스트]
        {text}
        """
        
        response = model.generate_content(prompt)
        return response.text
        
    except Exception as e:
        print(f"❌ Gemini 요약 오류: {e}")
        return None

def generate_output_path(media_path, is_summary=False):
    """파일 저장 경로 생성"""
    directory = os.path.dirname(media_path)
    base_name = os.path.splitext(os.path.basename(media_path))[0]
    date_str = datetime.now().strftime("%y%m%d")
    
    suffix = "summary" if is_summary else "script"
    filename = f"{base_name}_{date_str}_{suffix}.txt"
    
    return os.path.join(directory, filename)

def main():
    try:
        print(f"\n✨ AI 미디어 요약기 (Whisper {WHISPER_MODEL_NAME} + {GEMINI_MODEL_NAME})")
        print("=" * 70)
        
        media_path = select_media_file()
        if not media_path:
            print("파일 선택이 취소되었습니다.")
            return
        
        # 1. 텍스트 변환 (Whisper)
        text_path = generate_output_path(media_path, is_summary=False)
        full_text = transcribe_audio(media_path, text_path)
        
        if not full_text:
            return
            
        # 2. 요약 (Gemini)
        print(f"\n🧠 {GEMINI_MODEL_NAME} 모델이 내용을 분석 중입니다...")
        summary_text = summarize_text(full_text)
        
        if summary_text:
            summary_path = generate_output_path(media_path, is_summary=True)
            with open(summary_path, "w", encoding="utf-8") as f:
                f.write(summary_text)
            print(f"✅ 요약 완료! 파일 저장 위치:\n   -> {summary_path}")
            print("\n모든 작업이 성공적으로 끝났습니다.")
            
    except KeyboardInterrupt:
        print("\n사용자에 의해 프로그램이 중단되었습니다.")
    except Exception as e:
        print(f"\n예상치 못한 오류 발생: {e}")
    finally:
        # 종료 오류 메시지 숨기기 (stderr 리다이렉트)
        try:
            sys.stderr.close()
            sys.stderr = open(os.devnull, 'w')
        except:
            pass
        print("\n프로그램을 종료합니다.")

if __name__ == "__main__":
    main()